/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Protocol Definitions Function Header
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file ProtocolArchive.h
    \brief The API for the Protocol Archive.

 The protocol archive houses a number of protocols. Supported protocols include KEY, ERI and MOTPROG.
 The specific protocols supported by an archive is dependant upon its "flavour". Different archives exists to
 best cater for minimal footprint size versus required functionality. */

#ifndef __PROTOCOLARCHIVE_H
#define __PROTOCOLARCHIVE_H

/*---- INCLUDE FILES ----------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <MemFuncs.h>
#include <Resource.h>
#include <DFMProtocol.h>
#include <CANKing.h>
#include <ProtocolNotify.h>
#include <BlackBoxFuncs.h>

/*---- DEFINES ----------------------------------------------------------------------------------------------*/

/*---- TYPEDEF ----------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Describes the list of attributes associated with registering a protocol on a CAN capable resource.

    Each attribute corresponds to a member of the \ref S_RegisterCANProtocolInfo struture. Each valid attribute
    is logic-ORed together. Enabling attributes in this manner will allow the contents of the associated 
    structure to be extended without impacting on any existing code. */
typedef enum
{
    USE_PROTOCOL_ARCHIVE_DEFAULTS    = 0x00,
    USE_PROTOCOL_SECURITY            = 0x01, /*!< Affect S_RegisterCANProtocolInfo::pfProtocolSecurityCback and
                                                  S_RegisterCommProtocolInfo::pfSecurityCback */
    USE_CANKING_SECURITY_AUTHORITY   = 0x02, /*!< Affect S_RegisterCANProtocolInfo::pfAuthoritySecurityCback */
    USE_CANKING_CONNECTION           = 0x04, /*!< Affect S_RegisterProtocolArchiveInfo::pfConnectCback */
    USE_CANKING_CONTROL_REQUEST      = 0x08, /*!< Affect S_RegisterProtocolArchiveInfo::pfControlRequestCback */
    USE_CANKING_DATA_REQUEST         = 0x10, /*!< Affect S_RegisterProtocolArchiveInfo::pfDataRequestCback */
    USE_CANKING_ON_AUTHORITY_CHANGE  = 0x20, /*!< Affect S_RegisterProtocolArchiveInfo::pfAuthorityIncreasedCback */
    USE_ERI_DFM_HANDLER              = 0x40, /*!< Affect S_RegisterProtocolArchiveInfo::pfDiagnosticFunctionManager */
    USE_BLACK_BOX_REGISTRATION       = 0x80, /*!< Affect S_RegisterProtocolArchiveInfo::BlackBoxObj */

/*! Selects the S_RegisterProtocolArchiveInfo::u2KEYPktTimeout attribute. */
    USE_KEY_PKT_TIMEOUT              = 0x0100,

/*! Selects the S_RegisterProtocolArchiveInfo::u2ERIPktTimeout attribute. */
    USE_ERI_PKT_TIMEOUT              = 0x0200,

/* IF THIS TYPE EXCEEDS 0x8000 THEN ALTER THE TYPE OF THE RegisterProtocolArchiveAttributeMask_U ACCORDINGLY */

} E_RegisterProtocolArchiveAttributeMask;

/*! Unsigned integer type of sufficient size to hold the attribute masks for the protocol archive object atributes
    described by the \ref E_RegisterProtocolArchiveAttributeMask enumeration */
typedef uint2 RegisterProtocolArchiveAttributeMask_U;

/*! \brief The protocol archive's configuration attributes that define how the archive should start-up.

    Failure to correctly define acceptable values for these members will either result in the archive failing to
    register correctly, or unexpected behaviour. The S_RegisterProtocolArchiveInfo::uValidAttributesMask member
    has been provided to cater for later extension. Each optional attribute as an attribute mask assigned to it
    that should be set prior to making the call to register. */
typedef struct
{
/*! Identifies which attributes have values different from their assumed defaults. Member used to maintain
    backward compatibility with applications that may have been designed for use with an older version of
    this structure. The protocol archive may fail to register if some of the attributes are not defined */
    RegisterProtocolArchiveAttributeMask_U uValidAttributesMask;

/*! The scheduler resource that this archive is to use to service its periodic tasks. The archive will fail to
    register if this is not set to an appropriate value. */
    E_ModuleResource eSchedulerResource;

/*! The application's Diagnostic Function Manager (DFM). Any DFM requests that are received through an appropriate
    protocol will be dispatched to this manager. Set to \c NULL if the application does not support a diagnostic
    function manager.  Default is \c NULL. To apply this attribute include the \ref USE_ERI_DFM_HANDLER mask */
    DFMFuncPtr pfDiagnosticFunctionManager;

/*! The callback function that is executed when the authority level of a CANKING_PROTOCOL connection is
    successfully increased. The city that increased its authority along with the old and new authority is
    provided as parameters of the callback function. Set to \c NULL if the application does not have a handler.
    Default is \c NULL. To apply this attribute include the \ref USE_CANKING_ON_AUTHORITY_CHANGE mask. */
    CANKingAuthorityIncreasedCbackFuncPtr pfAuthorityIncreasedCback;

/*! The callback function that will be executed each time the authority level of a CANKING_PROTOCOL connection
    is successfully negotiated. Default is \c NULL. To apply this attribute include the 
    \ref USE_CANKING_CONNECTION mask */
    CANKingConnectCbackFuncPtr pfConnectCback;

/*! The callback function that will be executed when a CANKING_PROTOCOL data request is made. Set to \c NULL
    if the application does not have data request handler. Default is \c NULL. To apply this attribute include
    the \ref USE_CANKING_DATA_REQUEST mask. */
    CANKingDataRequestCbackFuncPtr pfDataRequestCback;

/*! The callback function that will be executed when a CANKING_PROTOCOL control request is made. Set to \c NULL
    if the application does not have a data request handler. Default is \c NULL. To apply this attribute include
    the \ref USE_CANKING_CONTROL_REQUEST mask. */
    CANKingControlRequestCbackFuncPtr pfControlRequestCback;

/*! The Black Box recorder protocol allows a remote tool to interrogate the module for specific information
    which can then be extracted. For this to work the application must expose hooks that can be executed when
    such remote requests are made. This object describes these hooks. Defaults as disabled. To apply this attribute
    include the \ref USE_BLACK_BOX_REGISTRATION mask in S_RegisterProtocolArchiveInfo::uValidAttributesMask */
    S_RegisterBlackBoxProperties BlackBoxObj;

/*! This attribute defines the maximum allowed time between the receipt of consecutive packets while the
    \ref KEY_PROTOCOL is in use before an exception is raised and the protocol stack is returned to the
    \ref KEY_PROTOCOL protocol and the access level reset back #1. The baud rate (where applicable for the
    transport in use) is set back to the default of 9600 BAUD.
    
    Set to zero to have this timeout ignored. A value of 2 seconds is a reasonable value for this attribute.

    This attribute has units of milliseconds.
    
    Select this attribute with the USE_KEY_PKT_TIMEOUT mask. If this attribute is not defined then it takes on
    the calibrated value described by KEYPktTimeout. KEYPktTimeout, when not included in the calibration, has the
    value of zero. */
    uint2 u2KEYPktTimeout;

/*! This attribute defines the maximum allowed time between the receipt of consecutive packets while the
    \ref ERI_PROTOCOL is in use before an exception is raised and the protocol stack is returned to the
    \ref KEY_PROTOCOL protocol and the access level reset back #1. The baud rate (where applicable for the
    transport in use) is set back to the default of 9600 BAUD.
    
    Set to zero to have this timeout ignored. A value of 2 seconds is a reasonable value for this attribute.

    This attribute has units of milliseconds.
    
    Select this attribute with the USE_ERI_PKT_TIMEOUT mask. If this attribute is not defined then it takes on
    the calibrated value described by ERIPktTimeout. ERIPktTimeout, when not included in the calibration, has the
    value of zero. */
    uint2 u2ERIPktTimeout;
} S_RegisterProtocolArchiveInfo;

typedef S_RegisterProtocolArchiveInfo const* S_RegisterProtocolArchiveInfoPtr;

/*! These are protocols that can be switched to from the \ref MOTOTRON_KEY_PROTOCOL. These protocols can not
    be directly registered through a RegisterCommProtocol() or RegisterCANProtocol() call. They are 
    peer-to-peer connections that require a security exchange (see \ref ProtocolSecurityCbackFuncPtr) to
    succeed before they can be switched to. */
typedef enum
{
    KEY_PROTOCOL = 0,           /*!< The KEY protocol. It is the same as the \ref MOTOTRON_KEY_PROTOCOL. */
    ERI_PROTOCOL = 1,           /*!< The ERI or ECM Remote Interface protocol. Used to access ECM data */
    MOTPROG_PROTOCOL = 3,       /*!< The MOTPROG protocol, which allows module programming */
    CANKING_PROTOCOL = 6,       /*!< CAN KING protocol which allows a peer-to-peer connection, typically with CAN */
    BLACKBOX_PROTOCOL = 7,      /*!< Simple data access protocol */
    MICROCHI_PROTOCOL = 8,      /*!< MicroCHI Protocol */
    SMARTREMOTE_PROTOCOL = 9,   /*!< Smart Remote Protocol */
    RUI_PROTOCOL = 10,          /*!< Remote User Interface protocol */
    J1939_CDRIF_DIAGMODE_PROTOCOL = 11, /* Used to switch from Application into Custom J1939_CDRIF protocol */
    WOODWARD_BOOT_PROTOCOL = 12, /*!< Used to Jump into Woodward Boot Code */
	KWP_PROTOCOL_ECU_PROGRAMMING_MODE = 13, /*!< Jump to Keyword Prototol in ECU_PROGRAMMING_MODE */
    NUM_OF_PROTOCOLS
} E_Protocols;        

/*! \brief Prototype of the the security function used by a protocol to alter its access level or to allow a
           switch to a different protocol */
typedef NativeBool (*ProtocolSecurityCbackFuncPtr)(E_AccessLevel in_eSwitchToAccessLevel, E_Protocols in_eSwitchToProtocol, uint4* inout_pu4SeedAndResult);

/*! The protocols supported by this archive. One of these protcols can be registered on a communications
    resource using the RegisterCommProtocol() call. */
typedef enum
{
/*! KEY protocol is used as the stepping stone in obtaining a more capable connection. The key aspect
    of this protocol is its ability to supports a security exchange which if successful allows the link's access
    level, enumerated by \ref E_AccessLevel, and the protocol, enumerated by \ref E_Protocols to be changed. For
    example, the security exchange required of this protocol must succeed before a remote service could establish
    an ERI connection. */
    MOTOTRON_KEY_PROTOCOL
} E_MotoTronProtocol;

/*! Data structure used to describe the attributes necessary to register a protocol for use on a
    communications link. */
typedef struct
{
/*! The protocol that is being registered for use */
    E_MotoTronProtocol eProtocol;
/*! The module resource with BEHAVIOUR_COMM capability that will be used by the protocol for communication */
    E_ModuleResource eCommResource;
/*! The security function that will be executed when a protocol security request is made. */
    ProtocolSecurityCbackFuncPtr pfSecurityCback;
/*! Identifies which attributes have values different from their assumed defaults. Member used to maintain
    backward compatibility with applications that may have been designed for use with an older version of
    this structure. The protocol archive may fail to register if some of the attributes are not defined */
    RegisterProtocolArchiveAttributeMask_U uValidAttributesMask;
} S_RegisterCommProtocolInfo;

/*! Constant pointer to the \ref S_RegisterCommProtocolInfo data structure */
typedef S_RegisterCommProtocolInfo const* S_RegisterCommProtocolInfoPtr;

/*! \brief Data structure used to describe the attributes necessary to register a protocol for use on a
    CAN communications link. It is assumed that the bus will have been previously created */
typedef struct
{
/*! The protocol that is being registered for use */
    E_MotoTronProtocol eProtocol;
/*! The module resource with BEHAVIOUR_TOUCAN capability that will be used by the protocol for communication */
    E_ModuleResource eCANResource;
/*! The TouCAN hardware buffer that can be used by this CAN BUS for transmission. This buffer can be shared
    with other transmission processes provided it has a queue attached to it that has sufficient depth to support
    this archives use of the transmission buffer with the other users */
    E_TouCANBuffer eTXBuffer;
/*! The TouCAN hardware buffer that can be used by this CAN BUS for exclusive reception. The protocol will fail
    to register if the supplied buffer is not free for exclusive use. The buffer should also have a dedicated mask
    associated with it. \c TOUCAN_BUFF15 is best suited for use by this buffer. */
    E_TouCANBuffer eRXBuffer;

/*! The callback function that will be executed when a protocol security request is made. Protocols will fail
    to negotiate security if this value is set \c NULL */
    ProtocolSecurityCbackFuncPtr pfProtocolSecurityCback;

/*! The callback function that will be executed when a \ref CANKING_PROTOCOL change of authority request is made
    by a connected city. The requested authority is supplied along with a pointer to a buffer that holds the
    seed and returns with the result. To apply include the \ref USE_CANKING_SECURITY_AUTHORITY mask */
    CANKingAuthoritySecurityCbackFuncPtr pfAuthoritySecurityCback;

/*! Identifes the city that an external connection is connecting to. It is this module's city. The city is
    a required attribute as it makes up part of the MessageID. */
    uint1 u1MyCity;

/*! Identifies which attributes have values different from their assumed defaults. Member used to maintain
    backward compatibility with applications that may have been designed for use with an older version of
    this structure. The protocol archive may fail to register if some of the attributes are not defined */
    RegisterProtocolArchiveAttributeMask_U uValidAttributesMask;

} S_RegisterCANProtocolInfo;

/*! Constant pointer to the \ref S_RegisterCANProtocolInfo data structure */
typedef S_RegisterCANProtocolInfo const* S_RegisterCANProtocolInfoPtr;

#pragma pack()
/*---- EXTERNALS --------------------------------------------------------------------------------------------*/

/*---- PROTOTYPES -------------------------------------------------------------------------------------------*/
NativeError_S RegisterMotoTronProtocolArchive(S_RegisterProtocolArchiveInfoPtr in_pArchiveRegisterObj);
NativeError_S RegisterCommProtocol(S_RegisterCommProtocolInfoPtr in_pProtocalRegisterObj);
NativeError_S RegisterCANProtocol(S_RegisterCANProtocolInfoPtr in_pProtocalRegisterObj);
NativeError_S SetCANProtocolCity(E_ModuleResource in_eCANResource, uint1 in_u1NewCity);

#endif /* __PROTOCOLARCHIVE_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/

